"use client";
import Link from "next/link";
import { Button } from "@/components/ui/button";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useEffect, useState } from "react";
import { useRouter } from "next/navigation";
import { useAuth } from "@/context/AuthUserContext";
import ErrorAlert from "@/components/shared/alert/error-alert";
import { Loader2 } from "lucide-react";
import GoogleSignIn from "@/components/authentication/google-signin";
import { Separator } from "@/components/ui/separator";
import { getSession } from "@/actions/auth-actions";
import { applicationUrls } from "@/constants/application-urls";
// import { firebaseAuth } from "@/lib/firebase";

export function Login() {
  const router = useRouter();
  const [error, setError] = useState(null);
  const {
    signInWithEmailAndPasswordHandler,
    loading,
    error: authError,
    authUser,
  } = useAuth();

  useEffect(() => {
    // Check for auth errors
    if (authError) {
      if (authError == "auth/invalid-credential") {
        setError("Invalid Credentials");
      } else {
        setError("An error occurred");
      }
    }
  }, [authError, loading]);

  useEffect(() => {
    // Redirect to dashboard if user is authenticated
    async function redirect() {
      const hasSession = await getSession();
      if (authUser && hasSession) {
        router.push(applicationUrls.dashboard.root);
      }
    }
    redirect();
  }, [authUser, router]);

  async function handleSubmit(event) {
    event.preventDefault();
    setError(null);
    const email = event.target.email.value;
    const password = event.target.password.value;

    // Validate the email
    if (!email) {
      setError("Email is required");
      return;
    }

    // Validate the password
    if (!password) {
      setError("Password is required");
      return;
    }

    // Sign in the user
    try {
      await signInWithEmailAndPasswordHandler(email, password);
    } catch (err) {
      console.log(err);
    }
  }

  return (
    <Card className="mx-auto max-w-sm">
      <CardHeader>
        <CardTitle className="text-xl">Welcome Back</CardTitle>
        <CardDescription>
          Enter your email below to login to your account
        </CardDescription>
      </CardHeader>
      <CardContent>
        <div className="grid gap-6">
          {/* SignIn with Google Button */}
          <GoogleSignIn />
          <Separator />
          {/* Email and Password Form */}
          <form onSubmit={handleSubmit} className="grid gap-6">
            {error && <ErrorAlert message={error} />}
            <div className="grid gap-2">
              <Label htmlFor="email">Email</Label>
              <Input
                id="email"
                type="email"
                placeholder="m@example.com"
                required
              />
            </div>
            <div className="grid gap-2">
              <div className="flex items-center">
                <Label htmlFor="password">Password</Label>
                <Link
                  href={applicationUrls.login.forgotPassword}
                  className="ml-auto inline-block text-sm underline"
                  prefetch={false}
                >
                  Forgot your password?
                </Link>
              </div>
              <Input id="password" type="password" required />
            </div>
            <Button
              type="submit"
              className="w-full"
              disabled={loading || authUser}
            >
              {loading && <Loader2 className="w-6 h-6 mr-2 animate-spin" />}
              <span>Login</span>
            </Button>
          </form>
        </div>
        <div className="mt-4 text-center text-sm">
          Don&apos;t have an account? Consider sign up
        </div>
      </CardContent>
    </Card>
  );
}
